<?php
/*
Plugin Name: WC Remote EZ Sync
Description: همگام‌سازی دوطرفه محصول، قیمت و موجودی بین دو سایت ووکامرس (ارسال/دریافت محصول، قیمت، موجودی + ارسال دسته‌جمعی + دکمه بروزرسانی از پیشخوان).
Version: 2.2.0
Author: حمیدرضا ابراهیمی
Text Domain: wc-remote-ez-sync
License: GPL2+
*/

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

if ( ! class_exists( 'WCREZ' ) ) :

class WCREZ {

    const NONCE             = 'wcrez_nonce';
    const OPT_REMOTE_BASE   = 'wcrez_remote_base';
    const OPT_REMOTE_TOKEN  = 'wcrez_remote_token';
    const OPT_PRICE_MARGIN  = 'wcrez_price_margin';
    const META_LAST_SYNC    = '_wcrez_last_sync';

    public function __construct() {
        // Settings
        add_action( 'admin_menu', [ $this, 'admin_menu' ] );

        // REST API
        add_action( 'rest_api_init', [ $this, 'register_routes' ] );

        // Product metabox
        add_action( 'add_meta_boxes', [ $this, 'add_product_metabox' ] );

        // Assets
        add_action( 'admin_enqueue_scripts', [ $this, 'enqueue_admin' ] );

        // AJAX handlers (تک‌محصول)
        add_action( 'wp_ajax_wcrez_push', [ $this, 'ajax_push' ] );
        add_action( 'wp_ajax_wcrez_pull', [ $this, 'ajax_pull' ] );
        add_action( 'wp_ajax_wcrez_push_price', [ $this, 'ajax_push_price' ] );
        add_action( 'wp_ajax_wcrez_pull_price', [ $this, 'ajax_pull_price' ] );

        // Bulk actions در لیست محصولات
        add_filter( 'bulk_actions-edit-product', [ $this, 'register_bulk_actions' ] );
        add_filter( 'handle_bulk_actions-edit-product', [ $this, 'handle_bulk_actions' ], 10, 3 );
        add_action( 'admin_notices', [ $this, 'bulk_admin_notice' ] );
    }

    /* ---------------- Settings Page ---------------- */

    public function admin_menu() {
        add_submenu_page(
            'woocommerce',
            __( 'Remote EZ Sync', 'wc-remote-ez-sync' ),
            __( 'Remote EZ Sync', 'wc-remote-ez-sync' ),
            'manage_woocommerce',
            'wc-remote-ez-sync',
            [ $this, 'settings_page' ]
        );
    }

    protected function run_bulk_price_stock_sync() {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return new WP_Error( 'wcrez_no_wc', __( 'WooCommerce not active.', 'wc-remote-ez-sync' ) );
        }

        $remote = get_option( self::OPT_REMOTE_BASE, '' );
        $token  = get_option( self::OPT_REMOTE_TOKEN, '' );

        if ( ! $remote || ! $token ) {
            return new WP_Error( 'wcrez_missing', __( 'Remote URL or token not set.', 'wc-remote-ez-sync' ) );
        }

        $args = [
            'post_type'      => 'product',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'fields'         => 'ids',
        ];
        $ids = get_posts( $args );

        $success = 0;
        $fail    = 0;

        if ( empty( $ids ) ) {
            return (object) [ 'success' => 0, 'fail' => 0 ];
        }

        foreach ( $ids as $id ) {
            $product = wc_get_product( $id );
            if ( ! $product ) {
                $fail++;
                continue;
            }
            $sku = $product->get_sku();
            if ( ! $sku ) {
                // اگر SKU نداشت، در این سناریو ازش عبور می‌کنیم
                $fail++;
                continue;
            }

            $url = rtrim( $remote, '/' ) . '/wp-json/wcrez/v1/price';

            $res = wp_remote_post( $url, [
                'headers' => [
                    'Content-Type'  => 'application/json',
                    'X-WCREZ-Token' => $token,
                ],
                'timeout' => 30,
                'body'    => wp_json_encode( [
                    'sku'  => $sku,
                    'mode' => 'get',
                ] ),
            ] );

            if ( is_wp_error( $res ) ) {
                $fail++;
                continue;
            }

            $code = wp_remote_retrieve_response_code( $res );
            $body = json_decode( wp_remote_retrieve_body( $res ), true );

            if ( ! ( $code >= 200 && $code < 300 ) || empty( $body ) ) {
                $fail++;
                continue;
            }

            // آپدیت قیمت و موجودی
            if ( isset( $body['regular_price'] ) ) {
                $product->set_regular_price( wc_clean( $body['regular_price'] ) );
            }
            if ( array_key_exists( 'sale_price', $body ) ) {
                $sale = $body['sale_price'];
                $product->set_sale_price( $sale === '' ? '' : wc_clean( $sale ) );
            }
            if ( array_key_exists( 'stock_quantity', $body ) ) {
                $product->set_manage_stock( true );
                $product->set_stock_quantity( wc_stock_amount( $body['stock_quantity'] ) );
            }
            if ( isset( $body['stock_status'] ) ) {
                $product->set_stock_status( sanitize_text_field( $body['stock_status'] ) );
            }

            $product->save();
            update_post_meta( $product->get_id(), self::META_LAST_SYNC, current_time( 'mysql' ) );
            $success++;
        }

        return (object) [
            'success' => $success,
            'fail'    => $fail,
        ];
    }

    public function settings_page() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            return;
        }

        $notice_html = '';

        // Save settings
        if ( isset( $_POST['wcrez_save'] ) && check_admin_referer( 'wcrez_save_settings', 'wcrez_nonce_settings' ) ) {
            $remote = isset( $_POST['wcrez_remote_base'] ) ? esc_url_raw( trim( $_POST['wcrez_remote_base'] ) ) : '';
            $token  = isset( $_POST['wcrez_remote_token'] ) ? sanitize_text_field( $_POST['wcrez_remote_token'] ) : '';
            $margin = isset( $_POST['wcrez_price_margin'] ) ? floatval( $_POST['wcrez_price_margin'] ) : 0;

            update_option( self::OPT_REMOTE_BASE, $remote );
            update_option( self::OPT_REMOTE_TOKEN, $token );
            update_option( self::OPT_PRICE_MARGIN, $margin );

            $notice_html .= '<div class="updated"><p>' . esc_html__( 'Settings saved.', 'wc-remote-ez-sync' ) . '</p></div>';
        }

        // Run bulk sync (button)
        if ( isset( $_POST['wcrez_sync_ps'] ) && check_admin_referer( 'wcrez_sync_ps', 'wcrez_nonce_sync' ) ) {
            $result = $this->run_bulk_price_stock_sync();

            if ( is_wp_error( $result ) ) {
                $notice_html .= '<div class="notice notice-error"><p>' . esc_html( $result->get_error_message() ) . '</p></div>';
            } else {
                $notice_html .= '<div class="notice notice-success"><p>' .
                    sprintf(
                        'بروزرسانی قیمت و موجودی انجام شد. موفق: %d ، ناموفق: %d',
                        intval( $result->success ),
                        intval( $result->fail )
                    )
                    . '</p></div>';
            }
        }

        $remote = get_option( self::OPT_REMOTE_BASE, '' );
        $token  = get_option( self::OPT_REMOTE_TOKEN, '' );
        $margin = get_option( self::OPT_PRICE_MARGIN, 0 );

        ?>
        <div class="wrap">
            <h1><?php esc_html_e( 'WC Remote EZ Sync Settings', 'wc-remote-ez-sync' ); ?></h1>

            <?php echo $notice_html; ?>

            <form method="post" style="margin-bottom:20px;">
                <?php wp_nonce_field( 'wcrez_save_settings', 'wcrez_nonce_settings' ); ?>

                <table class="form-table" role="presentation">
                    <tr>
                        <th scope="row"><label for="wcrez_remote_base"><?php esc_html_e( 'Remote site base URL', 'wc-remote-ez-sync' ); ?></label></th>
                        <td>
                            <input type="url" class="regular-text" id="wcrez_remote_base" name="wcrez_remote_base" value="<?php echo esc_attr( $remote ); ?>" placeholder="https://example.com">
                            <p class="description"><?php esc_html_e( 'آدرس سایت مقصدی که می‌خواهید محصولات را به آن بفرستید یا از آن بگیرید.', 'wc-remote-ez-sync' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcrez_remote_token"><?php esc_html_e( 'Shared secret token', 'wc-remote-ez-sync' ); ?></label></th>
                        <td>
                            <input type="text" class="regular-text" id="wcrez_remote_token" name="wcrez_remote_token" value="<?php echo esc_attr( $token ); ?>">
                            <p class="description"><?php esc_html_e( 'این توکن باید در هر دو سایت یکسان باشد. برای امنیت درخواست‌های REST استفاده می‌شود.', 'wc-remote-ez-sync' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row"><label for="wcrez_price_margin"><?php esc_html_e( 'Price margin % (optional)', 'wc-remote-ez-sync' ); ?></label></th>
                        <td>
                            <input type="number" step="0.01" id="wcrez_price_margin" name="wcrez_price_margin" value="<?php echo esc_attr( $margin ); ?>" style="width:100px;"> %
                            <p class="description"><?php esc_html_e( 'اگر مقدار مثبت وارد کنید، هنگام ارسال محصول، این درصد روی قیمت اعمال می‌شود (مثلاً 10 یعنی 10٪ گران‌تر در سایت مقصد).', 'wc-remote-ez-sync' ); ?></p>
                        </td>
                    </tr>
                </table>

                <p class="submit">
                    <button type="submit" name="wcrez_save" class="button-primary"><?php esc_html_e( 'Save changes', 'wc-remote-ez-sync' ); ?></button>
                </p>
            </form>

            <hr>

            <h2>بروزرسانی قیمت و موجودی از وبسایت مقابل</h2>
            <p>با زدن دکمه زیر، برای <strong>تمام محصولات این سایت</strong> که SKU دارند، قیمت و موجودی از وبسایت مقابل (بر اساس همین Remote URL و Token) دریافت و بروزرسانی می‌شود.</p>

            <form method="post">
                <?php wp_nonce_field( 'wcrez_sync_ps', 'wcrez_nonce_sync' ); ?>
                <p>
                    <button type="submit" name="wcrez_sync_ps" class="button button-secondary button-large">
                        بروزرسانی قیمت و موجودی وبسایت
                    </button>
                </p>
            </form>

            <h2><?php esc_html_e( 'How it works', 'wc-remote-ez-sync' ); ?></h2>
            <ol>
                <li><?php esc_html_e( 'این افزونه را در هر دو سایت نصب و فعال کنید.', 'wc-remote-ez-sync' ); ?></li>
                <li><?php esc_html_e( 'در هر دو سایت، در این صفحه، Remote URL و Token را تنظیم کنید (هرکدام به سایت مقابل اشاره کند و توکن مشترک باشد).', 'wc-remote-ez-sync' ); ?></li>
                <li><?php esc_html_e( 'برای ارسال/دریافت تک‌محصول، از باکس «Remote EZ Sync» در صفحه‌ی ویرایش محصول استفاده کنید.', 'wc-remote-ez-sync' ); ?></li>
                <li><?php esc_html_e( 'برای ارسال دسته‌جمعی محصولات، از گزینه «ارسال محصولات به وبسایت مقصد» در کارهای دسته‌جمعی استفاده کنید.', 'wc-remote-ez-sync' ); ?></li>
                <li><?php esc_html_e( 'برای بروزرسانی انبوه قیمت و موجودی، از دکمه بالا در همین صفحه استفاده کنید.', 'wc-remote-ez-sync' ); ?></li>
            </ol>
        </div>
        <?php
    }

    /* ---------------- REST API ---------------- */

    public function register_routes() {
        register_rest_route( 'wcrez/v1', '/push', [
            'methods'  => 'POST',
            'callback' => [ $this, 'rest_push' ],
            'permission_callback' => [ $this, 'rest_auth' ],
        ] );

        register_rest_route( 'wcrez/v1', '/pull', [
            'methods'  => 'POST',
            'callback' => [ $this, 'rest_pull' ],
            'permission_callback' => [ $this, 'rest_auth' ],
        ] );

        register_rest_route( 'wcrez/v1', '/price', [
            'methods'  => 'POST',
            'callback' => [ $this, 'rest_price' ],
            'permission_callback' => [ $this, 'rest_auth' ],
        ] );
    }

    public function rest_auth( $request ) {
        $token_header = $request->get_header( 'X-WCREZ-Token' );
        $opt_token    = get_option( self::OPT_REMOTE_TOKEN, '' );

        if ( empty( $opt_token ) || empty( $token_header ) || ! hash_equals( (string) $opt_token, (string) $token_header ) ) {
            return new WP_Error( 'wcrez_forbidden', __( 'Invalid token.', 'wc-remote-ez-sync' ), [ 'status' => 403 ] );
        }

        return true;
    }

    /**
     * دریافت کامل محصول از سایت فرستنده و ساخت/آپدیت در همین سایت (سمت مقصد).
     */
    public function rest_push( WP_REST_Request $request ) {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return new WP_Error( 'wcrez_no_wc', __( 'WooCommerce not active.', 'wc-remote-ez-sync' ), [ 'status' => 500 ] );
        }

        $data = $request->get_json_params();
        if ( empty( $data['sku'] ) ) {
            return new WP_Error( 'wcrez_no_sku', __( 'SKU is required.', 'wc-remote-ez-sync' ), [ 'status' => 400 ] );
        }

        $sku = sanitize_text_field( $data['sku'] );

        $product_id = wc_get_product_id_by_sku( $sku );
        if ( $product_id ) {
            $product = wc_get_product( $product_id );
        } else {
            // Create new product
            $post_arr = [
                'post_title'   => isset( $data['name'] ) ? sanitize_text_field( $data['name'] ) : $sku,
                'post_content' => isset( $data['description'] ) ? wp_kses_post( $data['description'] ) : '',
                'post_excerpt' => isset( $data['short_description'] ) ? wp_kses_post( $data['short_description'] ) : '',
                'post_status'  => isset( $data['status'] ) ? sanitize_text_field( $data['status'] ) : 'publish',
                'post_type'    => 'product',
            ];
            $product_id = wp_insert_post( $post_arr );
            if ( is_wp_error( $product_id ) ) {
                return $product_id;
            }
            $product = wc_get_product( $product_id );
            $product->set_sku( $sku );
        }

        $this->apply_product_array_to_product( $data, $product );
        $product->save();

        update_post_meta( $product->get_id(), self::META_LAST_SYNC, current_time( 'mysql' ) );

        return new WP_REST_Response( [
            'ok'  => true,
            'id'  => $product->get_id(),
            'sku' => $sku,
        ], 200 );
    }

    /**
     * برگرداندن اطلاعات کامل محصول (سمت مرجع) بر اساس SKU.
     */
    public function rest_pull( WP_REST_Request $request ) {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return new WP_Error( 'wcrez_no_wc', __( 'WooCommerce not active.', 'wc-remote-ez-sync' ), [ 'status' => 500 ] );
        }

        $data = $request->get_json_params();
        if ( empty( $data['sku'] ) ) {
            return new WP_Error( 'wcrez_no_sku', __( 'SKU is required.', 'wc-remote-ez-sync' ), [ 'status' => 400 ] );
        }

        $sku = sanitize_text_field( $data['sku'] );
        $product_id = wc_get_product_id_by_sku( $sku );
        if ( ! $product_id ) {
            return new WP_Error( 'wcrez_not_found', __( 'Product not found.', 'wc-remote-ez-sync' ), [ 'status' => 404 ] );
        }

        $product = wc_get_product( $product_id );
        $arr     = $this->build_product_array( $product );

        return new WP_REST_Response( $arr, 200 );
    }

    /**
     * فقط قیمت و موجودی – get/set.
     */
    public function rest_price( WP_REST_Request $request ) {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return new WP_Error( 'wcrez_no_wc', __( 'WooCommerce not active.', 'wc-remote-ez-sync' ), [ 'status' => 500 ] );
        }

        $data = $request->get_json_params();
        if ( empty( $data['sku'] ) ) {
            return new WP_Error( 'wcrez_no_sku', __( 'SKU is required.', 'wc-remote-ez-sync' ), [ 'status' => 400 ] );
        }

        $sku = sanitize_text_field( $data['sku'] );
        $product_id = wc_get_product_id_by_sku( $sku );
        if ( ! $product_id ) {
            return new WP_Error( 'wcrez_not_found', __( 'Product not found.', 'wc-remote-ez-sync' ), [ 'status' => 404 ] );
        }

        $product = wc_get_product( $product_id );
        $mode    = isset( $data['mode'] ) ? sanitize_text_field( $data['mode'] ) : 'get';

        if ( 'get' === $mode ) {
            return new WP_REST_Response( [
                'sku'            => $sku,
                'regular_price'  => $product->get_regular_price(),
                'sale_price'     => $product->get_sale_price(),
                'stock_quantity' => $product->get_stock_quantity(),
                'stock_status'   => $product->get_stock_status(),
            ], 200 );
        }

        if ( isset( $data['regular_price'] ) ) {
            $product->set_regular_price( wc_clean( $data['regular_price'] ) );
        }

        if ( array_key_exists( 'sale_price', $data ) ) {
            $sale = $data['sale_price'];
            $product->set_sale_price( $sale === '' ? '' : wc_clean( $sale ) );
        }

        if ( array_key_exists( 'stock_quantity', $data ) ) {
            $product->set_manage_stock( true );
            $product->set_stock_quantity( wc_stock_amount( $data['stock_quantity'] ) );
        }

        if ( isset( $data['stock_status'] ) ) {
            $product->set_stock_status( sanitize_text_field( $data['stock_status'] ) );
        }

        $product->save();
        update_post_meta( $product->get_id(), self::META_LAST_SYNC, current_time( 'mysql' ) );

        return new WP_REST_Response( [
            'ok'  => true,
            'id'  => $product->get_id(),
            'sku' => $sku,
        ], 200 );
    }

    /* ---------------- Helpers: build/apply product array ---------------- */

    protected function build_product_array( WC_Product $product ) {
        $data = [
            'id'                => $product->get_id(),
            'sku'               => $product->get_sku(),
            'name'              => $product->get_name(),
            'description'       => $product->get_description(),
            'short_description' => $product->get_short_description(),
            'status'            => $product->get_status(),
            'catalog_visibility'=> $product->get_catalog_visibility(),
            'regular_price'     => $product->get_regular_price(),
            'sale_price'        => $product->get_sale_price(),
            'manage_stock'      => $product->get_manage_stock(),
            'stock_quantity'    => $product->get_stock_quantity(),
            'stock_status'      => $product->get_stock_status(),
            'weight'            => $product->get_weight(),
            'length'            => $product->get_length(),
            'width'             => $product->get_width(),
            'height'            => $product->get_height(),
            'tax_status'        => $product->get_tax_status(),
            'tax_class'         => $product->get_tax_class(),
            'type'              => $product->get_type(),
            'categories'        => [],
            'images'            => [],
        ];

        // Categories by slug + name
        $terms = get_the_terms( $product->get_id(), 'product_cat' );
        if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
            foreach ( $terms as $t ) {
                $data['categories'][] = [
                    'slug' => $t->slug,
                    'name' => $t->name,
                ];
            }
        }

        // Images: main + gallery (URLs)
        $img_id = $product->get_image_id();
        if ( $img_id ) {
            $url = wp_get_attachment_url( $img_id );
            if ( $url ) {
                $data['images'][] = $url;
            }
        }
        $gallery_ids = $product->get_gallery_image_ids();
        if ( ! empty( $gallery_ids ) ) {
            foreach ( $gallery_ids as $gid ) {
                $url = wp_get_attachment_url( $gid );
                if ( $url && ! in_array( $url, $data['images'], true ) ) {
                    $data['images'][] = $url;
                }
            }
        }

        return $data;
    }

    protected function apply_product_array_to_product( array $data, WC_Product $product ) {
        if ( isset( $data['name'] ) ) {
            $product->set_name( wc_clean( $data['name'] ) );
        }
        if ( isset( $data['description'] ) ) {
            $product->set_description( wp_kses_post( $data['description'] ) );
        }
        if ( isset( $data['short_description'] ) ) {
            $product->set_short_description( wp_kses_post( $data['short_description'] ) );
        }
        if ( isset( $data['status'] ) ) {
            $product->set_status( sanitize_text_field( $data['status'] ) );
        }
        if ( isset( $data['catalog_visibility'] ) ) {
            $product->set_catalog_visibility( sanitize_text_field( $data['catalog_visibility'] ) );
        }

        if ( isset( $data['regular_price'] ) ) {
            $product->set_regular_price( wc_clean( $data['regular_price'] ) );
        }
        if ( array_key_exists( 'sale_price', $data ) ) {
            $product->set_sale_price( $data['sale_price'] === '' ? '' : wc_clean( $data['sale_price'] ) );
        }

        if ( isset( $data['manage_stock'] ) ) {
            $product->set_manage_stock( (bool) $data['manage_stock'] );
        }
        if ( isset( $data['stock_quantity'] ) ) {
            $product->set_stock_quantity( wc_stock_amount( $data['stock_quantity'] ) );
        }
        if ( isset( $data['stock_status'] ) ) {
            $product->set_stock_status( sanitize_text_field( $data['stock_status'] ) );
        }

        if ( isset( $data['weight'] ) ) {
            $product->set_weight( wc_clean( $data['weight'] ) );
        }
        if ( isset( $data['length'] ) ) {
            $product->set_length( wc_clean( $data['length'] ) );
        }
        if ( isset( $data['width'] ) ) {
            $product->set_width( wc_clean( $data['width'] ) );
        }
        if ( isset( $data['height'] ) ) {
            $product->set_height( wc_clean( $data['height'] ) );
        }
        if ( isset( $data['tax_status'] ) ) {
            $product->set_tax_status( sanitize_text_field( $data['tax_status'] ) );
        }
        if ( isset( $data['tax_class'] ) ) {
            $product->set_tax_class( sanitize_text_field( $data['tax_class'] ) );
        }

        // Categories
        if ( ! empty( $data['categories'] ) && is_array( $data['categories'] ) ) {
            $term_ids = [];
            foreach ( $data['categories'] as $cat ) {
                $slug = sanitize_title( $cat['slug'] ?? '' );
                $name = sanitize_text_field( $cat['name'] ?? $slug );
                if ( ! $slug ) {
                    continue;
                }
                $term = get_term_by( 'slug', $slug, 'product_cat' );
                if ( ! $term ) {
                    $result = wp_insert_term( $name, 'product_cat', [ 'slug' => $slug ] );
                    if ( ! is_wp_error( $result ) ) {
                        $term_ids[] = intval( $result['term_id'] );
                    }
                } else {
                    $term_ids[] = intval( $term->term_id );
                }
            }
            if ( ! empty( $term_ids ) ) {
                wp_set_object_terms( $product->get_id(), $term_ids, 'product_cat' );
            }
        }

        // Images: sideload and assign.
        if ( ! empty( $data['images'] ) && is_array( $data['images'] ) ) {
            $attachment_ids = [];
            foreach ( $data['images'] as $url ) {
                $url = esc_url_raw( $url );
                if ( ! $url ) {
                    continue;
                }

                // Try to find existing attachment by guid
                global $wpdb;
                $found_id = $wpdb->get_var( $wpdb->prepare(
                    "SELECT ID FROM {$wpdb->posts} WHERE guid = %s AND post_type = 'attachment' LIMIT 1",
                    $url
                ) );

                if ( $found_id ) {
                    $attachment_ids[] = intval( $found_id );
                    continue;
                }

                // Sideload
                if ( ! function_exists( 'media_sideload_image' ) ) {
                    require_once ABSPATH . 'wp-admin/includes/media.php';
                    require_once ABSPATH . 'wp-admin/includes/file.php';
                    require_once ABSPATH . 'wp-admin/includes/image.php';
                }

                $tmp = media_sideload_image( $url, 0, null, 'id' );
                if ( ! is_wp_error( $tmp ) && $tmp ) {
                    $attachment_ids[] = intval( $tmp );
                }
            }

            if ( ! empty( $attachment_ids ) ) {
                // First = main, rest = gallery
                $product->set_image_id( array_shift( $attachment_ids ) );
                $product->set_gallery_image_ids( $attachment_ids );
            }
        }
    }

    /* ---------------- Product Metabox ---------------- */

    public function add_product_metabox() {
        add_meta_box(
            'wcrez_metabox',
            __( 'Remote EZ Sync', 'wc-remote-ez-sync' ),
            [ $this, 'product_box' ],
            'product',
            'side',
            'high'
        );
    }

    public function product_box( WP_Post $post ) {
        if ( ! class_exists( 'WooCommerce' ) ) {
            echo '<p style="color:red;">' . esc_html__( 'WooCommerce must be active.', 'wc-remote-ez-sync' ) . '</p>';
            return;
        }

        wp_nonce_field( self::NONCE, self::NONCE );

        $remote = get_option( self::OPT_REMOTE_BASE, '' );
        $token  = get_option( self::OPT_REMOTE_TOKEN, '' );
        $margin = get_option( self::OPT_PRICE_MARGIN, 0 );

        $sku         = get_post_meta( $post->ID, '_sku', true );
        $last_sync   = get_post_meta( $post->ID, self::META_LAST_SYNC, true );
        $last_sync_t = $last_sync ? esc_html( $last_sync ) : __( 'Never', 'wc-remote-ez-sync' );

        echo '<p>' . esc_html__( 'همگام‌سازی سریع این محصول با سایت مقابل.', 'wc-remote-ez-sync' ) . '</p>';

        echo '<p><strong>SKU:</strong> ' . esc_html( $sku ? $sku : '(' . __( 'no SKU', 'wc-remote-ez-sync' ) . ')' ) . '</p>';
        echo '<p><strong>' . esc_html__( 'Remote URL:', 'wc-remote-ez-sync' ) . '</strong> ' . esc_html( $remote ? $remote : '-' ) . '</p>';
        echo '<p><strong>' . esc_html__( 'Last sync:', 'wc-remote-ez-sync' ) . '</strong> ' . $last_sync_t . '</p>';

        if ( $margin ) {
            echo '<p><strong>' . esc_html__( 'Price margin:', 'wc-remote-ez-sync' ) . '</strong> ' . esc_html( $margin ) . '%</p>';
        }

        echo '<input type="hidden" id="wcrez-remote" value="' . esc_attr( $remote ) . '">';
        echo '<input type="hidden" id="wcrez-token" value="' . esc_attr( $token ) . '">';

        echo '<button class="button button-primary" style="width:100%;margin-bottom:6px;" id="wcrez-send-full">' . esc_html__( 'ارسال کامل محصول به سایت مقابل', 'wc-remote-ez-sync' ) . '</button>';
        echo '<button class="button" style="width:100%;margin-bottom:12px;" id="wcrez-recv-full">' . esc_html__( 'دریافت کامل محصول از سایت مقابل', 'wc-remote-ez-sync' ) . '</button>';

        echo '<button class="button" style="width:100%;margin-bottom:6px;" id="wcrez-send-price">' . esc_html__( 'ارسال قیمت و موجودی', 'wc-remote-ez-sync' ) . '</button>';
        echo '<button class="button" style="width:100%;" id="wcrez-recv-price">' . esc_html__( 'دریافت قیمت و موجودی', 'wc-remote-ez-sync' ) . '</button>';

        echo '<div id="wcrez-msg" style="margin-top:10px;"></div>';
    }

    /* ---------------- Assets ---------------- */

    public function enqueue_admin( $hook ) {
        global $post;

        if ( ( $hook === 'post.php' || $hook === 'post-new.php' ) && isset( $post->post_type ) && $post->post_type === 'product' ) {
            wp_enqueue_script(
                'wcrez-admin',
                plugin_dir_url( __FILE__ ) . 'wcrez-admin.js',
                [ 'jquery' ],
                '2.2.0',
                true
            );

            wp_localize_script(
                'wcrez-admin',
                'WCREZ',
                [
                    'nonce' => wp_create_nonce( self::NONCE ),
                    'ajax'  => admin_url( 'admin-ajax.php' ),
                ]
            );
        }
    }

    /* ---------------- AJAX helpers & handlers (single product) ---------------- */

    protected function ajax_check() {
        if ( ! current_user_can( 'manage_woocommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'Forbidden.', 'wc-remote-ez-sync' ) ], 403 );
        }

        if ( ! isset( $_POST['nonce'] ) || ! wp_verify_nonce( $_POST['nonce'], self::NONCE ) ) {
            wp_send_json_error( [ 'message' => __( 'Invalid nonce.', 'wc-remote-ez-sync' ) ], 400 );
        }
    }

    /**
     * تابع کمکی که یک محصول را به سایت مقابل پوش می‌کند (تک محصول یا دسته‌جمعی).
     * اگر محصول SKU نداشته باشد، یک SKU خودکار مثل wcrez-{ID} برایش ست می‌کند.
     */
    protected function remote_push_single( $post_id, $remote = null, $token = null ) {
        if ( ! class_exists( 'WooCommerce' ) ) {
            return new WP_Error( 'wcrez_no_wc', __( 'WooCommerce not active.', 'wc-remote-ez-sync' ) );
        }

        $remote = $remote ? $remote : get_option( self::OPT_REMOTE_BASE, '' );
        $token  = $token  ? $token  : get_option( self::OPT_REMOTE_TOKEN, '' );
        $margin = floatval( get_option( self::OPT_PRICE_MARGIN, 0 ) );

        if ( ! $remote || ! $token ) {
            return new WP_Error( 'wcrez_missing', __( 'Remote URL or token not set.', 'wc-remote-ez-sync' ) );
        }

        $product = wc_get_product( $post_id );
        if ( ! $product ) {
            return new WP_Error( 'wcrez_not_found', __( 'Product not found.', 'wc-remote-ez-sync' ) );
        }

        $sku = $product->get_sku();
        if ( ! $sku ) {
            // ساخت SKU خودکار اگر وجود نداشت
            $sku = 'wcrez-' . $product->get_id();
            $product->set_sku( $sku );
            $product->save();
        }

        $data = $this->build_product_array( $product );

        // Apply margin on regular_price if set
        if ( $margin && isset( $data['regular_price'] ) && $data['regular_price'] !== '' ) {
            $base_price             = floatval( $data['regular_price'] );
            $data['regular_price']  = (string) ( $base_price * ( 1 + $margin / 100 ) );
        }

        $url = rtrim( $remote, '/' ) . '/wp-json/wcrez/v1/push';

        $res = wp_remote_post( $url, [
            'headers' => [
                'Content-Type'  => 'application/json',
                'X-WCREZ-Token' => $token,
            ],
            'timeout' => 60,
            'body'    => wp_json_encode( $data ),
        ] );

        if ( is_wp_error( $res ) ) {
            return $res;
        }

        $code = wp_remote_retrieve_response_code( $res );
        $body = json_decode( wp_remote_retrieve_body( $res ), true );

        if ( $code >= 200 && $code < 300 ) {
            update_post_meta( $product->get_id(), self::META_LAST_SYNC, current_time( 'mysql' ) );
            return true;
        }

        $msg = isset( $body['message'] ) ? $body['message'] : 'HTTP ' . $code;
        return new WP_Error( 'wcrez_http', $msg );
    }

    public function ajax_push() {
        $this->ajax_check();

        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        $remote  = isset( $_POST['remote'] ) ? esc_url_raw( $_POST['remote'] ) : '';
        $token   = isset( $_POST['token'] ) ? sanitize_text_field( $_POST['token'] ) : '';

        if ( ! $post_id ) {
            wp_send_json_error( [ 'message' => __( 'Missing product ID.', 'wc-remote-ez-sync' ) ] );
        }

        $result = $this->remote_push_single( $post_id, $remote, $token );

        if ( is_wp_error( $result ) ) {
            wp_send_json_error( [ 'message' => $result->get_error_message() ] );
        }

        wp_send_json_success( [ 'message' => __( 'محصول با موفقیت به سایت مقابل ارسال و ذخیره شد.', 'wc-remote-ez-sync' ) ] );
    }

    public function ajax_pull() {
        $this->ajax_check();

        if ( ! class_exists( 'WooCommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'WooCommerce not active.', 'wc-remote-ez-sync' ) ] );
        }

        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        $remote  = isset( $_POST['remote'] ) ? esc_url_raw( $_POST['remote'] ) : '';
        $token   = isset( $_POST['token'] ) ? sanitize_text_field( $_POST['token'] ) : '';

        if ( ! $post_id || ! $remote || ! $token ) {
            wp_send_json_error( [ 'message' => __( 'Missing data (post/remote/token).', 'wc-remote-ez-sync' ) ] );
        }

        $product = wc_get_product( $post_id );
        if ( ! $product ) {
            wp_send_json_error( [ 'message' => __( 'Product not found.', 'wc-remote-ez-sync' ) ] );
        }

        if ( ! $product->get_sku() ) {
            wp_send_json_error( [ 'message' => __( 'SKU is required for sync.', 'wc-remote-ez-sync' ) ] );
        }

        $url = rtrim( $remote, '/' ) . '/wp-json/wcrez/v1/pull';

        $res = wp_remote_post( $url, [
            'headers' => [
                'Content-Type'   => 'application/json',
                'X-WCREZ-Token'  => $token,
            ],
            'timeout' => 60,
            'body'    => wp_json_encode( [ 'sku' => $product->get_sku() ] ),
        ] );

        if ( is_wp_error( $res ) ) {
            wp_send_json_error( [ 'message' => $res->get_error_message() ] );
        }

        $code = wp_remote_retrieve_response_code( $res );
        $body = json_decode( wp_remote_retrieve_body( $res ), true );

        if ( ! ( $code >= 200 && $code < 300 ) || empty( $body ) || ! is_array( $body ) ) {
            $msg = isset( $body['message'] ) ? $body['message'] : 'HTTP ' . $code;
            wp_send_json_error( [ 'message' => $msg ] );
        }

        $this->apply_product_array_to_product( $body, $product );
        $product->save();
        update_post_meta( $product->get_id(), self::META_LAST_SYNC, current_time( 'mysql' ) );

        wp_send_json_success( [ 'message' => __( 'محصول از سایت مقابل دریافت و در این سایت به‌روزرسانی شد.', 'wc-remote-ez-sync' ) ] );
    }

    public function ajax_push_price() {
        $this->ajax_check();

        if ( ! class_exists( 'WooCommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'WooCommerce not active.', 'wc-remote-ez-sync' ) ] );
        }

        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        $remote  = isset( $_POST['remote'] ) ? esc_url_raw( $_POST['remote'] ) : '';
        $token   = isset( $_POST['token'] ) ? sanitize_text_field( $_POST['token'] ) : '';
        $margin  = floatval( get_option( self::OPT_PRICE_MARGIN, 0 ) );

        if ( ! $post_id || ! $remote || ! $token ) {
            wp_send_json_error( [ 'message' => __( 'Missing data (post/remote/token).', 'wc-remote-ez-sync' ) ] );
        }

        $product = wc_get_product( $post_id );
        if ( ! $product ) {
            wp_send_json_error( [ 'message' => __( 'Product not found.', 'wc-remote-ez-sync' ) ] );
        }

        if ( ! $product->get_sku() ) {
            wp_send_json_error( [ 'message' => __( 'SKU is required for sync.', 'wc-remote-ez-sync' ) ] );
        }

        $regular = $product->get_regular_price();
        $sale    = $product->get_sale_price();

        if ( $margin && $regular !== '' ) {
            $regular = (string) ( floatval( $regular ) * ( 1 + $margin / 100 ) );
        }

        $payload = [
            'sku'           => $product->get_sku(),
            'regular_price' => $regular,
            'sale_price'    => $sale,
            'stock_quantity'=> $product->get_stock_quantity(),
            'stock_status'  => $product->get_stock_status(),
            'mode'          => 'set',
        ];

        $url = rtrim( $remote, '/' ) . '/wp-json/wcrez/v1/price';

        $res = wp_remote_post( $url, [
            'headers' => [
                'Content-Type'   => 'application/json',
                'X-WCREZ-Token'  => $token,
            ],
            'timeout' => 30,
            'body'    => wp_json_encode( $payload ),
        ] );

        if ( is_wp_error( $res ) ) {
            wp_send_json_error( [ 'message' => $res->get_error_message() ] );
        }

        $code = wp_remote_retrieve_response_code( $res );
        $body = json_decode( wp_remote_retrieve_body( $res ), true );

        if ( $code >= 200 && $code < 300 ) {
            update_post_meta( $product->get_id(), self::META_LAST_SYNC, current_time( 'mysql' ) );
            wp_send_json_success( [ 'message' => __( 'قیمت و موجودی محصول در سایت مقابل به‌روزرسانی شد.', 'wc-remote-ez-sync' ) ] );
        }

        $msg = isset( $body['message'] ) ? $body['message'] : 'HTTP ' . $code;
        wp_send_json_error( [ 'message' => $msg ] );
    }

    public function ajax_pull_price() {
        $this->ajax_check();

        if ( ! class_exists( 'WooCommerce' ) ) {
            wp_send_json_error( [ 'message' => __( 'WooCommerce not active.', 'wc-remote-ez-sync' ) ] );
        }

        $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
        $remote  = isset( $_POST['remote'] ) ? esc_url_raw( $_POST['remote'] ) : '';
        $token   = isset( $_POST['token'] ) ? sanitize_text_field( $_POST['token'] ) : '';

        if ( ! $post_id || ! $remote || ! $token ) {
            wp_send_json_error( [ 'message' => __( 'Missing data (post/remote/token).', 'wc-remote-ez-sync' ) ] );
        }

        $product = wc_get_product( $post_id );
        if ( ! $product ) {
            wp_send_json_error( [ 'message' => __( 'Product not found.', 'wc-remote-ez-sync' ) ] );
        }

        if ( ! $product->get_sku() ) {
            wp_send_json_error( [ 'message' => __( 'SKU is required for sync.', 'wc-remote-ez-sync' ) ] );
        }

        $url = rtrim( $remote, '/' ) . '/wp-json/wcrez/v1/price';

        $res = wp_remote_post( $url, [
            'headers' => [
                'Content-Type'   => 'application/json',
                'X-WCREZ-Token'  => $token,
            ],
            'timeout' => 30,
            'body'    => wp_json_encode( [
                'sku'  => $product->get_sku(),
                'mode' => 'get',
            ] ),
        ] );

        if ( is_wp_error( $res ) ) {
            wp_send_json_error( [ 'message' => $res->get_error_message() ] );
        }

        $code = wp_remote_retrieve_response_code( $res );
        $body = json_decode( wp_remote_retrieve_body( $res ), true );

        if ( ! ( $code >= 200 && $code < 300 ) || empty( $body ) ) {
            $msg = isset( $body['message'] ) ? $body['message'] : 'HTTP ' . $code;
            wp_send_json_error( [ 'message' => $msg ] );
        }

        if ( isset( $body['regular_price'] ) ) {
            $product->set_regular_price( wc_clean( $body['regular_price'] ) );
        }
        if ( array_key_exists( 'sale_price', $body ) ) {
            $sale = $body['sale_price'];
            $product->set_sale_price( $sale === '' ? '' : wc_clean( $sale ) );
        }
        if ( array_key_exists( 'stock_quantity', $body ) ) {
            $product->set_manage_stock( true );
            $product->set_stock_quantity( wc_stock_amount( $body['stock_quantity'] ) );
        }
        if ( isset( $body['stock_status'] ) ) {
            $product->set_stock_status( sanitize_text_field( $body['stock_status'] ) );
        }

        $product->save();
        update_post_meta( $product->get_id(), self::META_LAST_SYNC, current_time( 'mysql' ) );

        wp_send_json_success( [ 'message' => __( 'قیمت و موجودی محصول از سایت مقابل دریافت و در این سایت به‌روزرسانی شد.', 'wc-remote-ez-sync' ) ] );
    }

    /* ---------------- Bulk actions (product list) ---------------- */

    public function register_bulk_actions( $bulk_actions ) {
        // متن گزینه در منوی کارهای دسته‌جمعی
        $bulk_actions['wcrez_send_products'] = 'ارسال محصولات به وبسایت مقصد';
        return $bulk_actions;
    }

    public function handle_bulk_actions( $redirect_to, $doaction, $post_ids ) {
        if ( $doaction !== 'wcrez_send_products' ) {
            return $redirect_to;
        }

        $remote = get_option( self::OPT_REMOTE_BASE, '' );
        $token  = get_option( self::OPT_REMOTE_TOKEN, '' );

        if ( ! $remote || ! $token ) {
            return add_query_arg( [
                'wcrez_bulk'        => 1,
                'wcrez_bulk_success'=> 0,
                'wcrez_bulk_fail'   => count( (array) $post_ids ),
                'wcrez_bulk_error'  => 'no_remote',
            ], $redirect_to );
        }

        $success = 0;
        $fail    = 0;

        foreach ( (array) $post_ids as $post_id ) {
            $result = $this->remote_push_single( $post_id, $remote, $token );
            if ( is_wp_error( $result ) ) {
                $fail++;
            } else {
                $success++;
            }
        }

        return add_query_arg( [
            'wcrez_bulk'         => 1,
            'wcrez_bulk_success' => $success,
            'wcrez_bulk_fail'    => $fail,
        ], $redirect_to );
    }

    public function bulk_admin_notice() {
        if ( ! isset( $_GET['wcrez_bulk'] ) ) {
            return;
        }

        $success = isset( $_GET['wcrez_bulk_success'] ) ? intval( $_GET['wcrez_bulk_success'] ) : 0;
        $fail    = isset( $_GET['wcrez_bulk_fail'] ) ? intval( $_GET['wcrez_bulk_fail'] ) : 0;
        $error   = isset( $_GET['wcrez_bulk_error'] ) ? sanitize_text_field( $_GET['wcrez_bulk_error'] ) : '';

        if ( $error === 'no_remote' ) {
            echo '<div class="notice notice-error is-dismissible"><p>برای استفاده از عملیات «ارسال محصولات به وبسایت مقصد» باید در تنظیمات افزونه، Remote URL و Token را تنظیم کنید.</p></div>';
            return;
        }

        echo '<div class="notice notice-success is-dismissible"><p>';
        printf(
            'ارسال دسته‌جمعی محصولات به وبسایت مقصد انجام شد. موفق: %d ، ناموفق: %d',
            $success,
            $fail
        );
        echo '</p></div>';
    }
}

new WCREZ();

endif;
